<?php
/**
 * Plugin Name:       Meloo Toolkit
 * Plugin URI:        http://rascalsthemes.com/
 * Description:       This is a Meloo Toolkit plugin for the Meloo WordPress theme. This plugin extends theme functionality. Is a required part of the theme.
 * Version:           1.0.2
 * Author:            Rascals Themes
 * Author URI:        http://rascalsthemes.com
 * Text Domain:       meloo-toolkit
 * License:           See "Licensing" Folder
 * License URI:       See "Licensing" Folder
 * Domain Path:       /languages
 */

// don't load directly
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists( 'MelooToolkit' ) ) {

	/**
	 * Main MelooToolkit Class
	 *
	 * Contains the main functions for MelooToolkit
	 *
	 * @package         MelooToolkit
	 * @author          Rascals Themes
	 * @copyright       Rascals Themes
 	 * @version       	1.0.2
	 */
	class MelooToolkit {

		/* instances
		  -------------------------------- */ 
		public $artists = null;
		public $music = null;
		public $gallery = null;
		public $videos = null;
		// 
		public $events = null;
		public $scamp_player = null;
		// 
		public $customizer = null;
		public $kc = null;
		public $shortcodes = null;
		public $admin_pages = null;
		public $importer = null;
		public $metaboxes = null;
		public $widgets = null;


		/* Other variables
		 -------------------------------- */
		
		// @var string
		public $theme_slug = 'meloo';

		// @var string
		public $theme_panel = 'meloo_panel_opts';

		// @var string
		public $version = '1.0.0';

		// @var integer
		public static $id = 0;

		// @var Single instance of the class
		private static $_instance;


		/**
		 * Meloo Toolkit Instance
		 *
		 * Ensures only one instance of Meloo Toolkit is loaded or can be loaded.
		 *
		 * @static
		 * @return Meloo Toolkit - Main instance
		 */
		public static function getInstance() {
			if ( ! ( self::$_instance instanceof self ) ) {
				self::$_instance = new self();
			}

			return self::$_instance;
		}

		/**
		 * Meloo Toolkit Constructor.
		 * @return void
		 */
		public function __construct() {

			if ( defined( 'RASCALS_TOOLKIT_PATH' ) ) {
				add_action( 'admin_notices', array( $this,  'noticeToolkitActivated' ) );
				return;
			}

			if ( get_option( 'rascals_toolkit' ) === false && get_option( 'rascals_toolkit' ) !== $this->theme_slug ) {
				add_action( 'admin_notices', array( $this,  'noticeOldTheme' ) );
				return;
			}

			// If old plugin is activated
			if ( in_array( 'rt-meloo-extensions/rt-meloo-extensions.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
				add_action( 'admin_notices', array( $this,  'noticeOldPlugin' ) );
				return;
			}

			// Get theme panel options
			$option_name = $this->theme_panel;
			$this->theme_panel = get_option( $option_name );

			$this->defineConstants();
			$this->initAdmin();
			$this->initTools();
			$this->initHooks();

			do_action( 'meloo_toolkit_loaded' );
		}

		/**
		 * Include required core files used in admin and on the backend.
		 * @return void
		 */
		public function initAdmin() {

			if ( ! is_admin() ) {
				return;
			}

			// Admin Pages
			include_once( $this->pluginPath() . '/includes/class-admin.php' );
			$this->admin_pages = new RascalsAdminPages();

			// Importer
			include_once( $this->pluginPath() . '/includes/importer/class-importer.php' );
			include_once( $this->pluginPath() . '/includes/importer/class-importer-data.php' );
			$this->importer = new RascalsImporterData;

			// Metaboxes
			include_once( $this->pluginPath() . '/includes/class-register-metaboxes.php' );
			$this->metaboxes = new RascalsRegisterMetaBoxes();
			
		}


		/**
		 * Include required core files used in admin and on the frontend.
		 * @return void
		 */
		public function initTools() {
		
			// Register custom post types
			include_once( $this->pluginPath() . '/includes/class-cpt.php' );

			include_once( $this->pluginPath() . '/includes/post-types/class-artists.php' );
			$this->artists = new RascalsRegisterArtists();

			include_once( $this->pluginPath() . '/includes/post-types/class-music.php' );
			$this->music = new RascalsRegisterMusic();

			include_once( $this->pluginPath() . '/includes/post-types/class-gallery.php' );
			$this->gallery = new RascalsRegisterGallery();

			include_once( $this->pluginPath() . '/includes/post-types/class-videos.php' );
			$this->videos = new RascalsRegisterVideos();

			include_once( $this->pluginPath() . '/includes/post-types/class-events-manager.php' );
			$this->events = new RascalsEventsManager();

			include_once( $this->pluginPath() . '/includes/scamp-player/class-scamp-player.php' );
			include_once( $this->pluginPath() . '/includes/scamp-player/shortcodes-scamp-player.php' );
			$this->scamp_player = new RascalsScampPlayer( array( 
				'theme_name' => 'meloo', 
				'theme_panel' => $this->theme_panel,
				'post_type' => 'meloo_tracks'
			) );

			// Twitter
			include_once( $this->pluginPath() . '/includes/class-twitter.php' );

			// Resize
			include_once( $this->pluginPath() . '/includes/class-image-resize.php' );

			// King Composer
			include_once( $this->pluginPath() . '/includes/class-kc.php' );
			$this->kc = new RascalsKC( array( 
				'theme_name'    => 'meloo', 
				'theme_panel'   => $this->theme_panel,
				'supported_cpt' => array( 'meloo_music', 'meloo_events', 'meloo_gallery', 'meloo_videos', 'meloo_artists' ),
				'default_fonts' => '{"Roboto":["cyrillic-ext%2Ccyrillic%2Cgreek%2Cvietnamese%2Clatin%2Clatin-ext%2Cgreek-ext","100%2C100italic%2C300%2C300italic%2Cregular%2Citalic%2C500%2C500italic%2C700%2C700italic%2C900%2C900italic","latin-ext","100%2C100italic%2C300%2C300italic%2Cregular%2Citalic%2C500%2C500italic%2C700%2C700italic%2C900%2C900italic"],"Barlow%20Condensed":["latin%2Clatin-ext","100%2C100italic%2C200%2C200italic%2C300%2C300italic%2Cregular%2Citalic%2C500%2C500italic%2C600%2C600italic%2C700%2C700italic%2C800%2C800italic%2C900%2C900italic","latin%2Clatin-ext","regular%2C500%2C600%2C700%2C800%2C900"],"Space%20Mono":["vietnamese%2Clatin%2Clatin-ext","regular%2Citalic%2C700%2C700italic","latin%2Clatin-ext","regular%2Citalic"]}'
			) );

			// Widgets
			include_once( $this->pluginPath() . '/includes/class-widgets.php' );
			add_action( 'widgets_init', array( $this, 'registerWidget' ) );
			$this->widgets = new RascalsWidgets();


			// Customizer Installer
			include_once( $this->pluginPath() . '/includes/class-customizer-install.php' );
			include_once( $this->pluginPath() . '/includes/class-customizer.php' );
			$this->customizer = new RascalsCustomizer();

			// Super menu
			require_once ABSPATH . 'wp-admin/includes/nav-menu.php';
			include_once( $this->pluginPath() . '/includes/super-menu-walker.php' );

			// Revolution Slider
			add_action( 'plugins_loaded', array( $this, 'setRevoSlider' ) );
			
		}


		/**
		 * Register widgets function.
		 *
		 * @return void
		 */
		public function registerWidget() {

			// Recent Posts
			include_once( $this->pluginPath() . '/includes/widgets/class-widget-recent-posts.php' );
			register_widget( 'RascalsRecentPostsWidget' );

			// Instagram feed
			include_once( $this->pluginPath() . '/includes/widgets/class-widget-instafeed.php' );
			register_widget( 'RascalsInstafeedWidget' );

			// Instagram feed
			include_once( $this->pluginPath() . '/includes/widgets/class-widget-ad.php' );
			register_widget( 'RascalsADWidget' );

		}


		/**
		 * Hook into actions and filters
		 * @return void
		 */
		private function initHooks() {

			add_action( 'admin_bar_menu', array( $this, 'showAdminBar' ), 100 );
			add_action( 'init', array( $this, 'init' ), 0 );
		}


 		/**
 		 * Init hooks when WordPress Initialises.
 		 * @return void
 		 */
		public function init() {

			// Before init action
			do_action( 'before_meloo_toolkit_init' );

			// Set localisation
			$this->loadPluginTextdomain();

			// Init action
			do_action( 'meloo_toolkit_init' );
		}


		/**
		 * Define constants
		 * @return void
		 */
		public function defineConstants() {

			// Plugin's directory path.
			define( 'RASCALS_TOOLKIT_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );

			// Plugin's directory URL.
			define( 'RASCALS_TOOLKIT_URL', untrailingslashit( plugins_url( '/', __FILE__ ) ) );

		}


		/**
		 * Loads the plugin text domain for translation
		 * @return void
		 */
		public function loadPluginTextdomain() {

			$domain = 'meloo-toolkit';
			$locale = apply_filters( 'meloo-toolkit', get_locale(), $domain );
			load_textdomain( $domain, WP_LANG_DIR . '/' . $domain . '/' . $domain . '-' . $locale . '.mo' );
			load_plugin_textdomain( $domain, FALSE, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
		}


		/**
		 * Display admin notice
		 * @return void
		 */
		public function noticeToolkitActivated(){
		    echo '<div class="notice rascals-notice notice-error is-dismissible">
		          <p><strong>' . esc_html__( 'Rascals Toolkit plugin is already activated, it was probably used by another Rascals Theme. Please dectivate old Toolkit plugin and activate new one.', 'meloo-toolkit') . '</strong></p>
		         </div>';
		}


		/**
		 * Display admin notice
		 * @return void
		 */
		public function noticeOldTheme(){
		    echo '<div class="notice rascals-notice notice-error is-dismissible">
		          <p><strong>' . esc_html__( 'Meloo Toolkit plugin is not compatible with installed theme. Please deactivate and activate your parent theme again or install compatible plugin with your theme version.', 'meloo-toolkit') . '</strong></p>
		         </div>';
		}


		/**
		 * Display admin notice
		 * @return void
		 */
		public function noticeOldPlugin(){
		    echo '<div class="notice rascals-notice notice-error is-dismissible">
		          <p><strong>' . esc_html__( 'Meloo Toolkit is currently disabled, please deactivate and remove Meloo Extensions plug-in to activate it. This is an outdated version and is not compatible with your theme.', 'meloo-toolkit') . '</strong></p>
		         </div>';
			
		}


		/**
		 * Get the plugin url.
		 * @return string
		 */
		public function pluginUrl() {
			return untrailingslashit( plugins_url( '/', __FILE__ ) );
		}


		/**
		 * Get the plugin path.
		 * @return string
		 */
		public function pluginPath() {
			return untrailingslashit( plugin_dir_path( __FILE__ ) );
		}


		/**
		 * Get the theme option
		 * @return string|bool|array
		 */
		public function get_theme_option( $option, $default = null ) {

			if ( $this->theme_panel === false || ! is_array( $this->theme_panel )  || ! isset( $option ) ) {
				return false;
			}
			if ( isset( $this->theme_panel[ $option ] ) ) {
				return $this->theme_panel[ $option ];
			} elseif ( $default !== null ) {
				return $default;
			} else {	
				return false;
			}
		
		}


		/**
		 * Display escaped text.
		 * @param  $text
		 * @return string
		 */
		public function esc( $text ) {
			$text = preg_replace( array('/<(\?|\%)\=?(php)?/', '/(\%|\?)>/'), array('',''), $text );
			return $text;
		}


		/**
		 * Display escaped text through echo function.
		 * @param  $text
		 * @return string
		 */
		public function e_esc( $text ) {
			echo preg_replace( array('/<(\?|\%)\=?(php)?/', '/(\%|\?)>/'), array('',''), $text );
		}


		/**
		 * Show admin bar hook
		 * @return void
		 */
		public function showAdminBar() {

			global $wp_admin_bar;
		
			if ( ! is_super_admin() || ! is_admin_bar_showing() ) {
				return;
			}

			$wp_admin_bar->add_menu(
				array( 
					'id'    => 'admin-welcome', 
					'title' => '<span class="ab-icon rascals-admin-link"></span> ' . esc_html__( 'Theme Settings', 'meloo-toolkit' ), 
					'href'  => get_bloginfo('wpurl') . '/wp-admin/admin.php?page=' . 'admin-welcome'
				)
			);
		}


		/**
		 * Set Revo Slider options for Ajax loader
		 * @version 1.0.0
		 * @return void
		 */
		public function setRevoSlider() {

			if ( class_exists( 'RevSlider' ) && function_exists( 'rev_slider_shortcode' ) ) {
				
				// Only for 6 and above versions
    			if ( defined('RS_REVISION') && version_compare( RS_REVISION, '6.0.0' ) >= 0 ) {
					$ajax = $this->get_theme_option( 'ajaxed', 'off' );

					if ( $ajax === 'on' ) {
						$rev_slider = new RevSlider();

						$rev_opts = $rev_slider->get_global_settings();
						if ( is_array( $rev_opts ) && isset( $rev_opts['include'] ) && $rev_opts['include'] === 'false' ) {
							$rev_opts['include'] = 'true';
							$rev_slider->set_global_settings($rev_opts);

						}
	            	}
	            }
        	}
        }


        /**
		 * Reize Image on fly and save in wp-content
		 * @return array|string
		 */
		static function imageResize( $url, $width = null, $height = null, $crop = null, $single = true, $upscale = false ) {
			/* WPML Fix */
	        if ( defined( 'ICL_SITEPRESS_VERSION' ) ){
	            global $sitepress;
	            $url = $sitepress->convert_url( $url, $sitepress->get_default_language() );
	        }
	        /* WPML Fix */
	        $rascalsResize = RascalsResize::getInstance();
	        return $rascalsResize->process( $url, $width, $height, $crop, $single, $upscale );
		}

	}
}


// Returns the main instance of MelooToolkit to prevent the need to use globals.
function melooToolkit() {
	return MelooToolkit::getInstance();
}

melooToolkit(); // Run