<?php
/**
 * Rascals Scamp Player
 *
 * @author Rascals Themes
 * @category Core
 * @package Meloo Toolkit
 * @version 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}


class RascalsScampPlayer {

	/*
	Private variables
	 */
	private $tracks_filter_args = array();

	/*
	Public variables
	 */
	public $theme_panel = null;
	public $post_type   = 'tracks';
	public $icon        = 'dashicons-format-audio';
	public $supports    = array('title', 'editor');


	/**
	 * Rascals CPT Constructor.
	 * @return void
	 */
	public function __construct( $options = null ) {

		if ( $options !== null ) {
			$this->theme_panel = $options['theme_panel'];
			$this->post_type = $options['post_type'];
		}
		$this->init();
	}

	/**
	 * Initialize class
	 * @return void
	 */
	public function init() {

		// Check if player is enabled
		if ( $this->option( 'scamp_player', 'off' ) === 'off' ) {
			return;
		}

		// Regiter Post type
		add_action( 'init', array( $this, 'regsiterPostType' ), 0 );

		// Add options to post type columns
		$this->addTracksColumns();

		// Frontend Scripts
		add_action( 'wp_enqueue_scripts' , array( $this, 'scampPlayerEnqueue' ) );

		// Add post state
		add_filter('display_post_states', array( $this, 'setPostState' ) );

		// Render Player
		add_action( 'wp_footer', array( $this, 'renderPlayer' ) );

	}


	/**
	 * Register Post Type
	 * @return void
	 */
	public function regsiterPostType() {

		// Class arguments 
	
		// Post arguments
		$post_options = array(
			'labels' => array(
				'name'               => esc_html__( 'Tracks Manager', 'meloo-toolkit' ),
				'singular_name'      => esc_html__( 'Track', 'meloo-toolkit' ),
				'add_new'            => esc_html__( 'Add New', 'meloo-toolkit' ),
				'add_new_item'       => esc_html__( 'Add New Tracks', 'meloo-toolkit' ),
				'edit_item'          => esc_html__( 'Edit Tracks', 'meloo-toolkit' ),
				'new_item'           => esc_html__( 'New Tracks', 'meloo-toolkit' ),
				'view_item'          => esc_html__( 'View Tracks', 'meloo-toolkit' ),
				'search_items'       => esc_html__( 'Search', 'meloo-toolkit' ),
				'not_found'          => esc_html__( 'No tracks found', 'meloo-toolkit' ),
				'not_found_in_trash' => esc_html__( 'No tracks found in Trash', 'meloo-toolkit' ), 
				'parent_item_colon'  => ''
			),
			'public'                => true,
			'show_ui'               => true,
			'show_in_nav_menus'     => true,
			'capability_typethis->' => 'post',
			'hierarchical'          => false,
			'rewrite'               => array(
				'slug'       => $this->option( 'tracks_slug', 'tracks' ),
				'with_front' => false
			),
			'supports'          => $this->supports,
			'menu_icon'         => $this->icon
		);

		$post_slug = $post_options['rewrite']['slug'];
		add_rewrite_rule('^'. $post_slug .'/page/([0-9]+)','index.php?pagename=artists&paged=$matches[1]', 'top');

		// Register Post Type 
		register_post_type( $this->post_type, $post_options );
		
	}


	/**
	 * Add extra columns to post table
	 * @param array $args
	 */
	public function addTracksColumns() {

		$this->tracks_filter_args = array(
			'post_name'    => $this->post_type,
			'extra_cols'   => array(
				'cb'            => '<input type="checkbox" />',
				'title'         => esc_html__( 'Title', 'meloo-toolkit' ),
				'preview'       => esc_html__( 'Preview', 'meloo-toolkit' ),
				'id'			=> esc_html__( 'Tracklist ID', 'meloo-toolkit' ),
			)
		);
		
		add_filter( 'manage_edit-' . esc_attr( $this->tracks_filter_args['post_name'] ) . '_columns', array( $this, 'filterTracksAddColumns' ) );
		add_filter( 'manage_posts_custom_column', array( $this, 'filterTracksDisplayColumns' ) );
		
	}


	/**
	 * Defined fields for columns
	 * @param  array $columns
	 * @return array 
	 */
	public function filterTracksAddColumns( $columns ) {

		// Get extra columns options
		$cols    = $this->tracks_filter_args['extra_cols'];
		$filters = $this->tracks_filter_args['filters'];
		$prefix  = $this->tracks_filter_args['post_name'] . '_';
		
		// Add unique ID to table columns
		foreach ( $cols as $i => $k ) {

			// ID
			if ( $i === 'id' ) {
				$cols[$prefix . 'id'] =  $cols['id'];
				unset( $cols['id'] );
			}

			// Preview
			if ( $i === 'preview' ) {
				$cols[$prefix . 'preview'] = $cols['preview'];
				unset( $cols['preview'] );
			}

		}


		// Merge extra columns
		$columns = array_merge( $columns, $cols );

		return $columns;
	}

	/**
	 * Display content in extra columns
	 * @param  array $column
	 * @return array
	 */
	public function filterTracksDisplayColumns( $column ) {

		global $post;

		$prefix  = $this->tracks_filter_args['post_name'] . '_';
		$cols    = $this->tracks_filter_args['extra_cols'];
		
		// Extra Cols
		foreach ( $cols as $i => $k ) {

			/* ID */
			if ( $i === 'id' && $column === $prefix . 'id' ) {
				echo esc_attr( $post->ID );
			}

			/* Preview */
			if ( $i === 'preview' && $column === $prefix . 'preview' ) {
				if ( has_post_thumbnail( $post->ID ) ) {
					the_post_thumbnail( array( 60, 60 ) );
				}
			}

		
		}
	}


	/**
	 * Enqueue all frontend scripts and styles 
	 * @return void
	 */
	public function scampPlayerEnqueue() {

		
		// Player Styles
		wp_enqueue_style( 'scamp-player' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/css/scamp.player.css' );

		if ( $this->option( 'player_skin' ) ) {
			wp_enqueue_style( 'scamp-player-skin' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/css/scamp.player.' . esc_attr( $this->option( 'player_skin' ) ) . '.css' );
		}


		// Player Scripts
		if ( $this->option( 'combine_js', 'off' ) === 'off' ) {

			// SC API
			if ( $this->option( 'soundcloud', 'on' ) === 'on' ) {
				wp_enqueue_script( 'soundcloud-sdk' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/vendors/sdk.js' , false, false, true );
			}

			// Shoutcast
			wp_enqueue_script( 'jquery-shoutcast' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/vendors/jquery.shoutcast.min.js' , false, false, true );
			// SM2
			wp_enqueue_script( 'soundmanager2' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/vendors/soundmanager2-nodebug-jsmin.js' , false, false, true );

			if ( $this->option( 'player_base64', 'on' ) === 'on' ) {
				wp_enqueue_script( 'base64' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/vendors/base64.min.js' , false, false, true );
			}

			// Scamp Player
			wp_enqueue_script( 'scamp-player' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/js/jquery.scamp.player.min.js' , false, false, true );
		} else {
			// SC API
			if ( $this->option( 'soundcloud', 'on' ) === 'on' ) {
				wp_enqueue_script( 'soundcloud-sdk' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/vendors/sdk.js' , false, false, true );
			}
			wp_enqueue_script( 'scamp-player' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/js/scamp-player-pack.min.js' , false, false, true );

		}

		// Shortcodes
		wp_enqueue_style( 'scamp-player-shortcodes' ,  esc_url( RASCALS_TOOLKIT_URL ) . '/assets/css/scamp-player-shortcodes.css' );

		// JS Init
	    wp_enqueue_script( 'scamp-player-init' , esc_url( RASCALS_TOOLKIT_URL ) . '/assets/js/jquery.scamp.player-init.js' , false, false, true );

	    $js_variables = array(
			'plugin_uri'         =>	esc_url( RASCALS_TOOLKIT_URL ) . '/assets/js/',
			'plugin_assets_uri'  => esc_url( RASCALS_TOOLKIT_URL ) . '/assets/',
			'shoutcast_interval' => $this->option( 'shoutcast_interval', 20000 ),
			'smooth_transition'  => false,
			'volume'             => $this->option( 'player_volume', '90' ),
			'play_label'         => esc_html__( 'Play', 'meloo-toolkit' ),
			'cover_label'        => esc_html__( 'Cover', 'meloo-toolkit' ),
			'title_label'        => esc_html__( 'Title/Artist', 'meloo-toolkit' ),
			'buy_label'          => esc_html__( 'Buy/Download', 'meloo-toolkit' ),
			'remove_label'       => esc_html__( 'Remove', 'meloo-toolkit' ),
			'empty_queue'        => esc_html__( 'Empty Queue', 'meloo-toolkit' )
		);

	    // Settings

		// Autoplay
		if ( $this->option( 'player_autoplay', 'off' ) === 'on' ) {
			$js_variables['autoplay'] = true;
		} else {
			$js_variables['autoplay'] = false;
		}

	 	// Base
		if ( $this->option( 'player_base64', 'on' ) === 'on' ) {
			$js_variables['base64'] = true;
		} else {
			$js_variables['base64'] = false;
		}

		// Random
		if ( $this->option( 'player_random', 'off' ) === 'on' ) {
			$js_variables['random'] = true;
		} else {
			$js_variables['random'] = false;
		}

		// Loop
		if ( $this->option( 'player_loop', 'on' ) === 'on' ) {
			$js_variables['loop'] = true;
		} else {
			$js_variables['loop'] = false;
		}

		// Load first track on tracklist
		if ( $this->option( 'load_first_track', 'on' ) === 'on' ) {
			$js_variables['load_first_track'] = true;
		} else {
			$js_variables['load_first_track'] = false;
		}

		// Show current track name instead off body title
		if ( $this->option( 'player_titlebar', 'on' ) === 'on' ) {
			$js_variables['titlebar'] = true;
		} else {
			$js_variables['titlebar'] = false;
		}

		// SC KEY
		if ( $this->option( 'soundcloud_id', '23f5c38e0aa354cdd0e1a6b4286f6aa4' ) !== '' ) {
			$js_variables['soundcloud_id'] = $this->option( 'soundcloud_id', '23f5c38e0aa354cdd0e1a6b4286f6aa4' );
		} else {
			$js_variables['soundcloud_id'] = '';
		}
	 
		// Shoutcast
		if ( $this->option( 'shoutcast', 'on' ) === 'on' ) {
			$js_variables['shoutcast'] = true;
		} else {
			$js_variables['shoutcast'] = false;
		}
		
		wp_localize_script( 'scamp-player-init', 'scamp_vars', $js_variables );

	
	}


	/**
	 * Get tracklist in array
	 * @param  integer $audio_post_id 
	 * @return array               
	 */
	public function getList( $audio_post, $custom_ids = '' ) {

		// Get IDS
		if ( $custom_ids !== '' ) {
			$audio_ids = $custom_ids;
		} else {
			$audio_ids = get_post_meta( $audio_post, '_audio_tracks', true );
		}

		if ( ! $audio_ids || $audio_ids === '' ) {
			 return false;
		}

		$count = 0;
		$ids = explode( '|', $audio_ids );
		$defaults = array(
			'id'               => 0,
			'custom'           => false,
			'custom_url'       => false,
			'title'            => '',
			'artists'          => false,
			'buttons'          => false,
			'cover'            => false,
			'cover_full'       => false,
			'release_url'      => '',
			'release_target'   => '', 
			'artists'          => '',
			'artists_url'      => '',
			'artists_target'   => '',
			'cart_url'         => '',
			'cart_target'      => '',
			'free_download'    => 'no',
			'track_length'     => '',
			'lyrics'           => '',
			'disable_playback' => 'no',
			'waveform'         => ''
		);

		$tracklist = array();

		/* Start Loop */
		foreach ( $ids as $id ) {

			// Vars 
			$title = '';
			$subtitle = '';

			// Get image meta 
			$track = get_post_meta( $audio_post, '_audio_tracks_' . $id, true );

			// Add default values 
			if ( isset( $track ) && is_array( $track ) ) {
				$track = array_merge( $defaults, $track );
			} else {
				$track = $defaults;
			}

			// ID 
			$track['id'] = $id;

			// Custom cover 
			if ( $track['cover'] ) {

				// If from Media Libary
				if ( is_numeric( $track['cover'] ) ) {
					$image_cover = wp_get_attachment_image_src( $track['cover'], 'thumbnail' );
					$image_cover = $image_cover[0];
					$image_cover_full = wp_get_attachment_image_src( $track['cover'], 'meloo-release-thumb' );
					$image_cover_full = $image_cover_full[0];
					if ( $image_cover ) {
						$track['cover'] =  $image_cover;
						$track['cover_full'] =  $image_cover_full;
					} else {
						$track['cover'] = false;
					}
				} else {
					$track['cover_full'] = $track['cover'];
				}

			}

			/* Waveform */
			if ( $track['waveform'] ) {

				$image_waveform = wp_get_attachment_image_src( $track['waveform'], 'full' );
				$image_waveform = $image_waveform[0];
			
				if ( $image_waveform ) {
					$track['waveform'] = $image_waveform;
				} else {
					$track['waveform'] = false;
				}
			}

			// Check if track is custom 
		   	if ( wp_get_attachment_url( $id ) ) {
		      	$track_att = get_post( $id );
		      	$track['url'] = wp_get_attachment_url( $id );
		      	if ( $track['title'] === '' ) {
		      		$track['title'] = $track_att->post_title;
		      	}
		    } else {
				$track['url'] = $track['custom_url'];
				if ( $track['url'] === '' ) {
					continue;
				}
				if ( $track['title'] === '' ) {
					$track['title'] = esc_html__( 'Custom Title', 'meloo-toolkit' );
				}
				$track['custom'] = true;
		    }
    
		    array_push( $tracklist, $track );
		}
		
		return $tracklist;
	
	}


	/**
	 * Add ScampPlayer container
	 * @return void
	 */
	public function renderPlayer() {

		// Player Classes
		$classes = '';

		// Open player after click on track
		if ( $this->option( 'open_player', 'on' ) === 'on' ) {
			$classes .= ' sp-open-after-click';
		}

		// Show Player on startup
		if ( $this->option( 'show_player', 'off' ) === 'on' ) {
			$classes .= ' sp-show-player';
		}

		// Show Tracklist on startup
		if ( $this->option( 'show_tracklist', 'off' ) === 'on' ) {
			$classes .= ' sp-show-list';
		}

		$classes .= ' sp-anim';
		?>
		<div id="scamp_player" class="<?php echo esc_attr( $classes ); ?>">
			<?php  
				// Startup Tracklist
				if ( $this->option( 'startup_tracklist', 'none' ) !== 'none' ) {
					$startup_tracklist = $this->getList( $this->option( 'startup_tracklist' ) );
					if ( $startup_tracklist ) {
						foreach ( $startup_tracklist as $track ) {
						
							if ( $track['disable_playback'] === 'no' ) {
								if ( $this->option( 'player_base64', 'off' ) === 'on' ) {
									$track['url'] = base64_encode( $track['url'] );
								}
								echo '<a href="' . $this->esc( $track['url'] ) . '" data-cover="' . esc_url( $track['cover'] ) . '" data-artist="' . esc_attr( $track['artists'] ) . '" data-artist_url="' . esc_url( $track['artists_url'] ) . '" data-artist_target="' . esc_attr( $track['artists_target'] ) . '" data-release_url="' . esc_url( $track['release_url'] ) . '" data-release_target="' . esc_attr( $track['release_target'] ) . '" data-shop_url="' . esc_url( $track['cart_url'] ) . '" data-shop_target="' . esc_attr( $track['cart_target'] ) . '" data-free_download="' . esc_attr( $track['free_download'] ) . '">' . esc_html( $track['title'] ) . '</a>';
							}
								
						}
					}
				}
			?>
		</div>
		<?php
		
	}


	/**
	 * Display autostart tracklist in post list
	 * @param array
	 */
	public function setPostState( $states ) { 
		global $post;
		
		if ( $this->post_type === get_post_type( $post->ID ) && $this->option( 'startup_tracklist' ) ) {
			$tracklist_id = $this->option( 'startup_tracklist' );
			if ( is_numeric( $tracklist_id ) ) {
				$tracklist_id = intval( $tracklist_id );

				if ( $tracklist_id === $post->ID ) {
					$states[] = esc_html__( 'Startup tracklist', 'meloo-toolkit' );
				}
			} 
			

		}

	    return $states;
	} 


	/**
	 * Get the theme option
	 * @return string|bool|array
	 */
	public function option( $option, $default = null ) {

		if ( $this->theme_panel !== null && is_array( $this->theme_panel ) ) {
	
			if ( isset( $this->theme_panel[ $option ] ) ) {
				return $this->theme_panel[ $option ];
			} elseif ( $default !== null ) {
				return $default;
			} else {	
				return false;
			}

		} elseif ( $default !== null ) {
			return $default;
		} else {
			return false;
		}
	}


	/**
	 * Display escaped text.
	 * @param  $text
	 * @return string
	 */
	public function esc( $text ) {
		$text = preg_replace( array('/<(\?|\%)\=?(php)?/', '/(\%|\?)>/'), array('',''), $text );
		return $text;
	}


	/**
	 * Display escaped text through echo function.
	 * @param  $text
	 * @return string
	 */
	public function e_esc( $text ) {
		echo preg_replace( array('/<(\?|\%)\=?(php)?/', '/(\%|\?)>/'), array('',''), $text );
	}

}